<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';
$page_title = 'New Return';
$db = db();

$customers = all($db, "SELECT id,name FROM customers ORDER BY name ASC");
$items = all($db, "SELECT id,name FROM items ORDER BY name ASC");

function next_return_no(mysqli $db): string {
  $row = one($db, "SELECT COALESCE(MAX(id),0)+1 AS n FROM returns");
  $n = (int)($row['n'] ?? 1);
  return 'RTN' . date('Ymd') . '-' . str_pad((string)$n, 4, '0', STR_PAD_LEFT);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check();
  $customer_id = (int)($_POST['customer_id'] ?? 0);
  $invoice_id = (int)($_POST['invoice_id'] ?? 0);
  if ($invoice_id<=0) $invoice_id = null;
  $return_no = trim($_POST['return_no'] ?? '');
  $return_date = $_POST['return_date'] ?? date('Y-m-d');
  $method = trim($_POST['method'] ?? 'Credit'); // Credit reduces invoice/customer balance; Cash does not
  $notes = trim($_POST['notes'] ?? '');

  $line_item_id = $_POST['item_id'] ?? [];
  $line_uom_id  = $_POST['uom_id'] ?? [];
  $line_qty     = $_POST['qty'] ?? [];
  $line_price   = $_POST['price'] ?? [];

  if ($return_no === '') $return_no = next_return_no($db);

  if ($customer_id<=0) {
    flash_set('danger','Select customer.');
  } else {
    $db->begin_transaction();
    try {
      $uid = (int)($_SESSION['user_id'] ?? 0);
      $total = 0.0;

      $st = $db->prepare("INSERT INTO returns(return_no,return_date,customer_id,invoice_id,method,total,notes,created_by) VALUES(?,?,?,?,?,?,?,?)");
      $zero=0.0;
      $st->bind_param("ssissdsi", $return_no, $return_date, $customer_id, $invoice_id, $method, $zero, $notes, $uid);
      $st->execute();
      $return_id = $st->insert_id;
      $st->close();

      $ins = $db->prepare("INSERT INTO return_items(return_id,item_id,uom_id,qty,price,line_total,cost_total) VALUES(?,?,?,?,?,?,?)");

      for ($i=0; $i<count($line_item_id); $i++) {
        $item_id = (int)$line_item_id[$i];
        $uom_id  = (int)($line_uom_id[$i] ?? 0);
        $qty     = (float)($line_qty[$i] ?? 0);
        $price   = (float)($line_price[$i] ?? 0);
        if ($item_id<=0 || $uom_id<=0 || $qty<=0) continue;

        $u = one($db, "SELECT price, conversion_qty FROM item_uoms WHERE id=? AND item_id=?","ii",[$uom_id,$item_id]);
        if ($u) $price = (float)$u['price'];
        $conv = (float)($u['conversion_qty'] ?? 1); if($conv<=0) $conv=1;

        $lt = $qty * $price;
        $total += $lt;

        // cost based on current avg cost
        $on = one($db, "SELECT avg_cost_base FROM stock_onhand WHERE item_id=?","i",[$item_id]);
        $avg_cost_base = (float)($on['avg_cost_base'] ?? 0);
        $cost_total = ($qty * $conv) * $avg_cost_base;

        $ins->bind_param("iiidddd", $return_id, $item_id, $uom_id, $qty, $price, $lt, $cost_total);
        $ins->execute();

        // stock back in
        $qty_base = $qty * $conv;
        $db->query("INSERT INTO stock_onhand(item_id, qty_base, avg_cost_base) VALUES ($item_id, 0, 0)
                    ON DUPLICATE KEY UPDATE item_id=item_id");
        $db->query("UPDATE stock_onhand SET qty_base = qty_base + ".((float)$qty_base)." WHERE item_id = $item_id");
        $db->query("INSERT INTO stock_moves(move_date,move_type,ref_type,ref_id,item_id,qty_base,cost_base,note)
                    VALUES (NOW(),'RETURN','return',$return_id,$item_id,".((float)$qty_base).",".((float)$avg_cost_base).",'Return')");
      }
      $ins->close();

      $up = $db->prepare("UPDATE returns SET total=? WHERE id=?");
      $up->bind_param("di",$total,$return_id);
      $up->execute(); $up->close();

      // If method is Credit: reduce invoice balance (or customer outstanding)
      if ($method === 'Credit') {
        if ($invoice_id) {
          $db->query("UPDATE invoices SET balance = GREATEST(balance - ".((float)$total).", 0) WHERE id = ".((int)$invoice_id)." AND customer_id = ".((int)$customer_id));
          $db->query("
            UPDATE invoices
            SET status = CASE
              WHEN balance <= 0.0001 THEN 'Paid'
              WHEN balance < total THEN 'Partial'
              ELSE 'Unpaid'
            END
            WHERE id = ".((int)$invoice_id)."
          ");
        } else {
          // No invoice linked: create a negative payment allocation concept is heavy; for MVP just store return and treat in reports
        }
      }

      $db->commit();
      flash_set('success','Return saved.');
      header('Location: ' . base_url() . '/public/return_view.php?id=' . $return_id);
      exit;
    } catch (Throwable $e) {
      $db->rollback();
      flash_set('danger','Failed: '.$e->getMessage());
    }
  }
}

$return_no = next_return_no($db);

include __DIR__ . '/../includes/header.php';
?>
<div class="card p-3">
  <div class="d-flex justify-content-between align-items-center">
    <h4 class="mb-0">New return</h4>
    <a class="btn btn-outline-light" href="<?= base_url() ?>/public/returns.php">Back</a>
  </div>

  <form method="post" class="mt-3" id="retForm">
    <?= csrf_field() ?>
    <div class="row g-2">
      <div class="col-12 col-md-5">
        <label class="form-label">Customer</label>
        <select class="form-select" name="customer_id" required>
          <option value="">Select customer</option>
          <?php foreach($customers as $c): ?>
            <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-6 col-md-3">
        <label class="form-label">Date</label>
        <input class="form-control" type="date" name="return_date" value="<?= h(date('Y-m-d')) ?>" required>
      </div>
      <div class="col-6 col-md-4">
        <label class="form-label">Return no</label>
        <input class="form-control" name="return_no" value="<?= h($return_no) ?>">
      </div>
    </div>

    <div class="row g-2 mt-1">
      <div class="col-12 col-md-4">
        <label class="form-label">Method</label>
        <select class="form-select" name="method">
          <option value="Credit">Credit (reduce balance)</option>
          <option value="Cash">Cash refund (no balance change)</option>
        </select>
      </div>
      <div class="col-12 col-md-4">
        <label class="form-label">Link to invoice (optional)</label>
        <input class="form-control" name="invoice_id" placeholder="Invoice ID (optional)">
        <div class="small opacity-75">Tip: open invoice view and copy the ID from URL.</div>
      </div>
      <div class="col-12 col-md-4">
        <label class="form-label">Notes</label>
        <input class="form-control" name="notes" placeholder="optional">
      </div>
    </div>

    <hr class="border-secondary my-3">

    <div class="table-responsive">
      <table class="table table-sm align-middle" id="lines">
        <thead>
          <tr>
            <th style="min-width:220px;">Item</th>
            <th style="min-width:180px;">UOM</th>
            <th class="text-end" style="width:120px;">Qty</th>
            <th class="text-end" style="width:140px;">Price</th>
            <th class="text-end" style="width:160px;">Line Total</th>
            <th style="width:60px;"></th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    </div>

    <div class="d-flex gap-2">
      <button type="button" class="btn btn-outline-info" onclick="addRow()">+ Add line</button>
      <button class="btn btn-primary ms-auto">Save return</button>
    </div>

    <div class="mt-3 text-end">
      <div class="opacity-75 small">Return total</div>
      <div class="fs-3 fw-bold" id="grandTotal">KES 0.00</div>
    </div>
  </form>
</div>

<script>
const ITEMS = <?= json_encode($items, JSON_UNESCAPED_UNICODE) ?>;
function money(n){ return (Number(n)||0).toLocaleString(undefined,{minimumFractionDigits:2,maximumFractionDigits:2}); }
function escapeHtml(s){ return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[m])); }

function addRow(){
  const tbody = document.querySelector('#lines tbody');
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td>
      <select class="form-select form-select-sm itemSel" name="item_id[]" required>
        <option value="">Select item</option>
        ${ITEMS.map(it=>`<option value="${it.id}">${escapeHtml(it.name)}</option>`).join('')}
      </select>
    </td>
    <td>
      <select class="form-select form-select-sm uomSel" name="uom_id[]" required disabled>
        <option value="">Select UOM</option>
      </select>
    </td>
    <td><input class="form-control form-control-sm text-end qty" name="qty[]" type="number" step="0.001" value="1" min="0.001" required></td>
    <td><input class="form-control form-control-sm text-end price" name="price[]" type="number" step="0.01" value="0" min="0"></td>
    <td class="text-end fw-semibold lineTotal">KES 0.00</td>
    <td class="text-end"><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calc();">×</button></td>
  `;
  tbody.appendChild(tr);

  const itemSel = tr.querySelector('.itemSel');
  const uomSel = tr.querySelector('.uomSel');
  itemSel.addEventListener('change', async () => {
    uomSel.innerHTML = '<option value="">Loading...</option>';
    uomSel.disabled = true;
    const itemId = itemSel.value;
    if(!itemId){ uomSel.innerHTML='<option value="">Select UOM</option>'; return; }
    const res = await fetch('<?= base_url() ?>/public/api/uoms.php?item_id=' + encodeURIComponent(itemId));
    const data = await res.json();
    const uoms = (data && data.uoms) ? data.uoms : [];
    uomSel.innerHTML = '<option value="">Select UOM</option>' + uoms.map(u=>`<option value="${u.id}" data-price="${u.price}">${escapeHtml(u.uom_name)}</option>`).join('');
    uomSel.disabled = false;
  });

  uomSel.addEventListener('change', () => {
    const opt = uomSel.options[uomSel.selectedIndex];
    const price = opt ? opt.getAttribute('data-price') : '0';
    tr.querySelector('.price').value = price || 0;
    calc();
  });

  tr.querySelector('.qty').addEventListener('input', calc);
  tr.querySelector('.price').addEventListener('input', calc);
  calc();
}

function calc(){
  let grand = 0;
  document.querySelectorAll('#lines tbody tr').forEach(tr=>{
    const qty = Number(tr.querySelector('.qty').value)||0;
    const price = Number(tr.querySelector('.price').value)||0;
    const lt = qty * price;
    grand += lt;
    tr.querySelector('.lineTotal').textContent = 'KES ' + money(lt);
  });
  document.getElementById('grandTotal').textContent = 'KES ' + money(grand);
}
addRow();
</script>
<?php include __DIR__ . '/../includes/footer.php'; ?>
