<?php
require_once __DIR__ . '/helpers.php';

/**
 * Safaricom Daraja STK Push helper (no composer).
 * Configure in config/app.php -> 'mpesa' section.
 */

function mpesa_cfg(): array {
  $cfg = app_cfg();
  return $cfg['mpesa'] ?? [];
}

function mpesa_base_url(): string {
  $c = mpesa_cfg();
  return (($c['environment'] ?? 'sandbox') === 'production')
    ? 'https://api.safaricom.co.ke'
    : 'https://sandbox.safaricom.co.ke';
}

function mpesa_http_post_json(string $url, array $payload, array $headers = []): array {
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_POST, true);
  curl_setopt($ch, CURLOPT_HTTPHEADER, array_merge([
    'Content-Type: application/json',
    'Accept: application/json',
  ], $headers));
  curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
  curl_setopt($ch, CURLOPT_TIMEOUT, 40);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($body === false) return ['ok'=>false,'code'=>$code,'error'=>$err,'raw'=>null];
  $json = json_decode($body, true);
  return ['ok'=>($code >= 200 && $code < 300), 'code'=>$code, 'json'=>$json, 'raw'=>$body];
}

function mpesa_access_token(): array {
  $c = mpesa_cfg();
  $key = trim((string)($c['consumer_key'] ?? ''));
  $sec = trim((string)($c['consumer_secret'] ?? ''));
  if ($key === '' || $sec === '' || $key === 'CHANGE_ME') {
    return ['ok'=>false, 'error'=>'M-Pesa consumer key/secret not set in config/app.php'];
  }

  $url = mpesa_base_url() . '/oauth/v1/generate?grant_type=client_credentials';
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
  curl_setopt($ch, CURLOPT_USERPWD, $key . ':' . $sec);
  curl_setopt($ch, CURLOPT_TIMEOUT, 40);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($body === false) return ['ok'=>false, 'code'=>$code, 'error'=>$err];
  $json = json_decode($body, true);
  if (!is_array($json) || empty($json['access_token'])) {
    return ['ok'=>false, 'code'=>$code, 'error'=>'Failed to get access token', 'raw'=>$body];
  }
  return ['ok'=>true, 'token'=>$json['access_token'], 'expires_in'=>($json['expires_in'] ?? null)];
}

function mpesa_password(string $shortcode, string $passkey, string $timestamp): string {
  return base64_encode($shortcode . $passkey . $timestamp);
}

function mpesa_stk_push(array $args): array {
  $c = mpesa_cfg();

  $shortcode = (string)($c['shortcode'] ?? '');
  $passkey   = (string)($c['passkey'] ?? '');
  $callback  = (string)($c['callback_url'] ?? '');
  $acc_ref   = (string)($c['account_reference'] ?? 'WholesaleOps');
  $desc      = (string)($c['transaction_desc'] ?? 'POS Payment');

  $phone  = preg_replace('/\s+/', '', (string)($args['phone'] ?? ''));
  $amount = (int)round((float)($args['amount'] ?? 0));
  if ($amount <= 0) return ['ok'=>false,'error'=>'Amount must be > 0'];
  if ($callback === '' && !empty($args['callback_url'])) $callback = (string)$args['callback_url'];

  if ($shortcode === '' || $passkey === '' || $passkey === 'CHANGE_ME') {
    return ['ok'=>false,'error'=>'M-Pesa shortcode/passkey not set in config/app.php'];
  }
  if ($callback === '') {
    return ['ok'=>false,'error'=>'M-Pesa callback_url not set in config/app.php'];
  }

  $phone = preg_replace('/[^0-9]/', '', $phone);
  if (str_starts_with($phone, '0')) $phone = '254' . substr($phone, 1);
  if (!str_starts_with($phone, '254')) return ['ok'=>false,'error'=>'Phone must be Kenyan format (07.. or 2547..)'];

  $tok = mpesa_access_token();
  if (!$tok['ok']) return $tok;

  $timestamp = date('YmdHis');
  $password = mpesa_password($shortcode, $passkey, $timestamp);

  $payload = [
    "BusinessShortCode" => $shortcode,
    "Password" => $password,
    "Timestamp" => $timestamp,
    "TransactionType" => "CustomerPayBillOnline",
    "Amount" => $amount,
    "PartyA" => $phone,
    "PartyB" => $shortcode,
    "PhoneNumber" => $phone,
    "CallBackURL" => $callback,
    "AccountReference" => ($args['account_reference'] ?? $acc_ref),
    "TransactionDesc" => ($args['transaction_desc'] ?? $desc),
  ];

  $url = mpesa_base_url() . "/mpesa/stkpush/v1/processrequest";
  $res = mpesa_http_post_json($url, $payload, [
    "Authorization: Bearer " . $tok['token']
  ]);

  if (!$res['ok']) {
    return ['ok'=>false,'error'=>'STK push failed','http_code'=>$res['code'],'response'=>($res['json'] ?? null),'raw'=>($res['raw'] ?? null)];
  }
  return ['ok'=>true,'response'=>$res['json']];
}
